#include "engine/PathFinder.h"

#include <cmath>
#include <list>

PathFinder::PathFinder()
{
}

PathFinder::~PathFinder()
{
}

float heuristicDistance(PathFinderNode const &pStartNode, PathFinderNode const &pEndNode)
{
    sf::Vector2f lDelta = pStartNode.position - pEndNode.position;
    return std::sqrtf(lDelta.x * lDelta.x + lDelta.y * lDelta.y);
}

int PathFinder::addNode(sf::Vector2f const &pNodePosition)
{
    PathFinderNode lNewNode;
    lNewNode.position = pNodePosition;
    mGraph.push_back(lNewNode);
    return mGraph.size() - 1;
}

void PathFinder::addNodeNeighbour(int pNodeId, int pNeighbourId)
{
    mGraph[pNodeId].neighbours.push_back(
        std::pair<PathFinderNode*, float>(
            &(mGraph[pNeighbourId]),
            heuristicDistance(mGraph[pNodeId], mGraph[pNeighbourId])));
}

void PathFinder::resetAStarData()
{
    for(unsigned int i = 0; i < mGraph.size(); ++i)
        mGraph[i].isInClosedList = mGraph[i].isInOpenList = false;
}

struct PathFinderOpenListNode
{
    PathFinderNode *graphNode;
    float cost;
    float heuristicCost;
};

bool operator<(PathFinderOpenListNode const& pLt, PathFinderOpenListNode const& pRt)
{
    return (pLt.cost + pLt.heuristicCost) > (pRt.cost + pRt.heuristicCost);
}

float PathFinder::computePath(int pStartNodeId, int pEndNodeId, std::vector<sf::Vector2f> &oPath)
{
    std::list<sf::Vector2f> lTemporaryPath;
    float lReturnValue = PathFinder::computePath(pStartNodeId, pEndNodeId, lTemporaryPath);
    oPath.clear();
	oPath.insert(oPath.begin(), lTemporaryPath.begin(), lTemporaryPath.end());
    return lReturnValue;
}

float PathFinder::computePath(int pStartNodeId, int pEndNodeId, std::list<sf::Vector2f> &oPath)
{
    PathFinderNode &pStartNode = mGraph[pStartNodeId];
    PathFinderNode &pEndNode = mGraph[pEndNodeId];

    resetAStarData();
    
	std::priority_queue<PathFinderOpenListNode> lOpenList;

	pStartNode.parent = 0;
	PathFinderOpenListNode lOpenListStartNode;
	lOpenListStartNode.graphNode = &pStartNode;
	lOpenListStartNode.cost = 0.0f;
	lOpenListStartNode.heuristicCost = heuristicDistance(pStartNode, pEndNode);
	lOpenList.push(lOpenListStartNode);
    
	while(!lOpenList.empty())
	{
		PathFinderOpenListNode lCurrentNode = lOpenList.top();
		lOpenList.pop();
        
		if (lCurrentNode.graphNode == &pEndNode)
		{
            float lHeuristicCost = lOpenListStartNode.heuristicCost;
			reconstructPath(pEndNode, oPath);
			return pEndNode.bestCost;
		}

		while(lCurrentNode.graphNode->isInClosedList)
		{
			if(lOpenList.empty())
				return -1.0f;
			lCurrentNode = lOpenList.top();
			lOpenList.pop();
		}
        
		lCurrentNode.graphNode->isInClosedList = true;
        
        for (unsigned int i = 0; i < lCurrentNode.graphNode->neighbours.size(); ++i)
		{
            PathFinderNode* lNeighbour = lCurrentNode.graphNode->neighbours[i].first;
			float lTotalDistance = lCurrentNode.cost + lCurrentNode.graphNode->neighbours[i].second;
			if (!lNeighbour->isInClosedList && (!lNeighbour->isInOpenList || (lTotalDistance < lNeighbour->bestCost)))
			{
				lNeighbour->isInOpenList = true;
				PathFinderOpenListNode lNewOpenListNode;
				lNewOpenListNode.graphNode = lNeighbour;
				lNeighbour->bestCost = lNewOpenListNode.cost = lTotalDistance;
				lNeighbour->parent = lCurrentNode.graphNode;
				lNewOpenListNode.heuristicCost = heuristicDistance(*lNeighbour, pEndNode);
				lOpenList.push(lNewOpenListNode);
			}
		}
	}
    return -1.0f;
}

sf::Vector2f PathFinder::nextPositionOnPath(int pStartNodeId, int pEndNodeId)
{
    PathFinderNode &pStartNode = mGraph[pStartNodeId];
    PathFinderNode &pEndNode = mGraph[pEndNodeId];

    resetAStarData();
    
	std::priority_queue<PathFinderOpenListNode> lOpenList;

	pStartNode.parent = NULL;
	PathFinderOpenListNode lOpenListStartNode;
	lOpenListStartNode.graphNode = &pStartNode;
	lOpenListStartNode.cost = 0.0f;
	lOpenListStartNode.heuristicCost = heuristicDistance(pStartNode, pEndNode);
	lOpenList.push(lOpenListStartNode);
    
	while(!lOpenList.empty())
	{
		PathFinderOpenListNode lCurrentNode = lOpenList.top();
		lOpenList.pop();
        
		if (lCurrentNode.graphNode == &pEndNode)
		{
            if(lCurrentNode.graphNode->parent)
                return lCurrentNode.graphNode->parent->position;
            else
                return pStartNode.position;
		}

		while(lCurrentNode.graphNode->isInClosedList)
		{
			if(lOpenList.empty())
                return pStartNode.position;
			lCurrentNode = lOpenList.top();
			lOpenList.pop();
		}
        
		lCurrentNode.graphNode->isInClosedList = true;
        
        for (unsigned int i = 0; i < lCurrentNode.graphNode->neighbours.size(); ++i)
		{
            PathFinderNode* lNeighbour = lCurrentNode.graphNode->neighbours[i].first;
			float lTotalDistance = lCurrentNode.cost + lCurrentNode.graphNode->neighbours[i].second;
			if (!lNeighbour->isInClosedList && (!lNeighbour->isInOpenList || (lTotalDistance < lNeighbour->bestCost)))
			{
				lNeighbour->isInOpenList = true;
				PathFinderOpenListNode lNewOpenListNode;
				lNewOpenListNode.graphNode = lNeighbour;
				lNeighbour->bestCost = lNewOpenListNode.cost = lTotalDistance;
                if(lCurrentNode.graphNode->parent == NULL)
                    lNeighbour->parent = lNeighbour;
                else
                    lNeighbour->parent = lCurrentNode.graphNode->parent;
				lNewOpenListNode.heuristicCost = heuristicDistance(*lNeighbour, pEndNode);
				lOpenList.push(lNewOpenListNode);
			}
		}
	}
    return pStartNode.position;
}

void PathFinder::reconstructPath(PathFinderNode &pEndNode, std::list<sf::Vector2f> &oPath)
{
    oPath.clear();
    oPath.push_front(pEndNode.position);
    PathFinderNode* lCurrentNode = &pEndNode;
    while (lCurrentNode = lCurrentNode->parent)
        oPath.push_front(lCurrentNode->position);
}
