#include "platformer/PlatformerScreen.h"

#include <dirent.h>

#include "Config.h"
#include "engine/GameTime.h"

#include "platformer/Player.h"
#include "platformer/Platform.h"
#include "platformer/Enemy.h"
#include "platformer/Collectible.h"
#include "platformer/PlatformerCollisionLayers.h"
#include "home/HomeScreen.h"
#include "home/WinScreen.h"

PlatformerScreen::PlatformerScreen(std::unique_ptr<Network> pNetwork, float pScreenWidth, float pScreenHeight):
    AScreen(),
    mScreenWidth(pScreenWidth),
    mScreenHeight(pScreenHeight),
    mNetwork(std::move(pNetwork)),
	mTimeInfoSend(TIME_SEND_INFO)
{
    mView.setCenter(0.5f * mScreenWidth, -0.5f * mScreenHeight);


    float lLevelHeight = (Config::PlatformerEnemy::maxHeight - 1) * 0.25f * Config::PlatformerPlayer::noLegJumpImpulse;
    
    std::vector<sf::String> lImageFilenames;
	struct dirent *ent;
	DIR *lDir = opendir ("data/platformer/backgrounds/");

    if (lDir != NULL) 
    {
        while ((ent = readdir (lDir)) != NULL) 
        {
            if(ent->d_type == DT_REG)
            {
                sf::String lStr(ent->d_name);
                if(lStr.find(".png") != sf::String::InvalidPos || lStr.find(".jpg") != sf::String::InvalidPos)
                    lImageFilenames.push_back(lStr);
            }
        }
        closedir (lDir);
    }

    for(int y = -Config::Graphics::height; y >= lLevelHeight - 2.0f * Config::Graphics::height; y -= Config::Graphics::height)
    {
        Sprite* lBackgroundSprite = new Sprite();
        lBackgroundSprite->loadFrom(sf::String("data/platformer/backgrounds/") + lImageFilenames[rand() % lImageFilenames.size()]);
        lBackgroundSprite->setPosition(sf::Vector2f(0, y));
        lBackgroundSprite->getSFSprite().setScale(
            (float)Config::Graphics::width / lBackgroundSprite->getSFSprite().getLocalBounds().width,
            (float)Config::Graphics::height / lBackgroundSprite->getSFSprite().getLocalBounds().height);
        mSprites.push_back(lBackgroundSprite);
    }
    lLevelHeight = (Config::PlatformerEnemy::maxHeight - 1) * 0.25f * Config::PlatformerPlayer::noLegJumpImpulse;
    Sprite* lGoalSprite = new Sprite();
    lGoalSprite->loadFrom("data/platformer/goal.png");
    lGoalSprite->setPosition(sf::Vector2f(0.5f * mScreenWidth - 0.5f * lGoalSprite->getSFSprite().getLocalBounds().width,
                                          lLevelHeight - lGoalSprite->getSFSprite().getLocalBounds().height));
    mSprites.push_back(lGoalSprite);
    
    for(int i = 1; i < Config::PlatformerEnemy::maxHeight; ++i)
    {
        lLevelHeight = i * 0.25f * Config::PlatformerPlayer::noLegJumpImpulse;
        if(i != (Config::PlatformerEnemy::maxHeight - 1))
        {
            float lProb = (float)rand() / (float)RAND_MAX;
            Platform* lNewPlatform = new Platform(lLevelHeight, mScreenWidth);
            mSprites.push_back(lNewPlatform);
            mCollisionManager.addBoundingBox(PlatformerCollisionLayers::Platform, lNewPlatform);
        
            if((rand() % 4) == 0)
            {
                Enemy* lEnemy = new Enemy(lNewPlatform, this, mSprites.size());
                mSprites.push_back(lEnemy);
                mCollisionManager.addBoundingBox(PlatformerCollisionLayers::Enemy, lEnemy);
            }
        }
        else
            for (int i = 0 ; i * 118.0f <  Config::Graphics::width ; ++i)
            {
                Platform* lNewPlatform = new Platform(sf::Vector2f(i * 118.0f, lLevelHeight));
                mSprites.push_back(lNewPlatform);
                mCollisionManager.addBoundingBox(PlatformerCollisionLayers::Platform, lNewPlatform);
            }
    }
    for (int i = 0 ; i * 118.0f <  Config::Graphics::width ; ++i)
    {
        Platform* lFirstPlatforms = new Platform(sf::Vector2f(i * 118.0f, -18.f));
        mSprites.push_back(lFirstPlatforms);
        mCollisionManager.addBoundingBox(PlatformerCollisionLayers::FirstPlatform, lFirstPlatforms);
    }

    std::vector<std::pair<std::string, std::pair<int, int> > > lCollectibles;
    lCollectibles.push_back(std::pair<std::string, std::pair<int, int> >("data/platformer/Collectibles/fruit1.png", std::pair<int, int>(3, -10)));
    lCollectibles.push_back(std::pair<std::string, std::pair<int, int> >("data/platformer/Collectibles/fruit2.png", std::pair<int, int>(3, -10)));
    lCollectibles.push_back(std::pair<std::string, std::pair<int, int> >("data/platformer/Collectibles/fruit3.png", std::pair<int, int>(3, -10)));
    lCollectibles.push_back(std::pair<std::string, std::pair<int, int> >("data/platformer/Collectibles/fat1.png", std::pair<int, int>(10, 10)));
    lCollectibles.push_back(std::pair<std::string, std::pair<int, int> >("data/platformer/Collectibles/fat2.png", std::pair<int, int>(10, 10)));
    lCollectibles.push_back(std::pair<std::string, std::pair<int, int> >("data/platformer/Collectibles/fat3.png", std::pair<int, int>(10, 10)));
    lCollectibles.push_back(std::pair<std::string, std::pair<int, int> >("data/platformer/Collectibles/fat1.png", std::pair<int, int>(10, 10)));
    lCollectibles.push_back(std::pair<std::string, std::pair<int, int> >("data/platformer/Collectibles/fat2.png", std::pair<int, int>(10, 10)));
    lCollectibles.push_back(std::pair<std::string, std::pair<int, int> >("data/platformer/Collectibles/fat3.png", std::pair<int, int>(10, 10)));
    
    for(int i = 0; i < Config::PlatformerEnemy::commestibles; ++i)
    {
        float lHProb = (float)rand() / (float)RAND_MAX;
        float lVProb = (float)rand() / (float)RAND_MAX;

        std::pair<std::string, std::pair<int, int> > const &lCollectibleParams = lCollectibles[rand() % lCollectibles.size()];
        Collectible* lNewCollectible =
            new Collectible(lCollectibleParams.first, lCollectibleParams.second.first, lCollectibleParams.second.second, this, mSprites.size());
        mCollisionManager.addBoundingBox(PlatformerCollisionLayers::Collectible, lNewCollectible);
        
        lNewCollectible->setPosition(
            sf::Vector2f(
                lHProb * (pScreenWidth - lNewCollectible->getSFSprite().getLocalBounds().width),
                lVProb * lLevelHeight - 64.0f));
        mSprites.push_back(lNewCollectible);
    }

    mSprites.push_back(mPlayer = new Player(sf::Vector2f(0.5f * mScreenWidth, -256.0f), mScreenWidth, mCollisionManager, *this));

    mMusic.openFromFile("data/platformer/ggj_prog_team_up.ogg");
    mMusic.setLoop(true);
    mMusic.setVolume(12);
    mMusic.play();

    mVictorySoundBuffer.loadFromFile("data/platformer/win.ogg");
    mVictorySound.setBuffer(mVictorySoundBuffer);
}

PlatformerScreen::~PlatformerScreen(void)
{
    for(unsigned int i = 0; i < mSprites.size(); ++i)
        if(mSprites[i])
            delete mSprites[i];
    mMusic.stop();
}

void PlatformerScreen::update()
{
    if (mNetwork.get())
    {
        Network& lNetwork = *mNetwork;
        Network::Message lMsg;
        while (lNetwork.getMessage(lMsg))
        {
            if (lMsg.isForPlateformerPlayer())
            {
                if (lMsg.isMemberKill())
                {
                    mPlayer->killMember(lMsg.getMember()); 
                }
                else if (lMsg.isMemberLife())
                {
					mPlayer->updateLife(lMsg.getMember(), lMsg.getLife());
                }
                else if (lMsg.isWin())
                {
                    ScreenManager::getInstance()->popScreen();
                    ScreenManager::getInstance()->pushScreen(new WinScreen(std::move(mNetwork), false));
                    return;
                }
            }
        }
		mTimeInfoSend -= Time::deltaTime();
		if (mTimeInfoSend <= 0)
		{
			mTimeInfoSend = TIME_SEND_INFO;
			float lLevelHeight = (Config::PlatformerEnemy::maxHeight - 1) * 0.25f * Config::PlatformerPlayer::noLegJumpImpulse - mPlayer->getSFSprite().getLocalBounds().height;
			float height = mPlayer->getPosition().y / lLevelHeight;
			if (height < 0) height = 0;
			if (height > 1) height = 1;
			lNetwork.platformerSendHeight(height);
		}
        if (lNetwork.isDisconnected())
        {
            ScreenManager::getInstance()->popScreen();
            ScreenManager::getInstance()->pushScreen(new HomeScreen(Config::Graphics::width, Config::Graphics::height));
            return;
        }
    }
    for(unsigned int i = 0; i < mSprites.size(); ++i)
        if(mSprites[i])
            mSprites[i]->update();

    mCollisionManager.checkCollisions(sf::Vector2i(0, 0));
    float lNewViewY = std::min(-0.5f * mScreenHeight, mPlayer->getPosition().y);
    lNewViewY = 0.99f* mView.getCenter().y + 0.01f * lNewViewY;
    mView.setCenter(0.5f * mScreenWidth, lNewViewY);
    
    if(mPlayer->mVictoryCoolDown > 0.0f)
    {
        mPlayer->mVictoryCoolDown -= Time::deltaTime();
        if(mPlayer->mVictoryCoolDown <= 0.0f)
        {
            onVictory();
            return;
        }
    }
    else
    {
        float lLevelHeight = (Config::PlatformerEnemy::maxHeight - 1) * 0.25f * Config::PlatformerPlayer::noLegJumpImpulse - mPlayer->getSFSprite().getLocalBounds().height;
        if(mPlayer->getPosition().y < lLevelHeight)
        {
            mPlayer->mVictoryCoolDown = 5.0f;
            mVictorySound.play();
        }
    }
}

void PlatformerScreen::draw(sf::RenderTarget* pTarget)
{
	pTarget->clear(sf::Color(0, 0, 0, 255));
    for(unsigned int i = 0; i < mSprites.size(); ++i)
        if(mSprites[i])
            mSprites[i]->draw(pTarget);
#if _DEBUG
    //mCollisionManager.drawDebug(*pTarget, sf::Color(0, 0, 255, 128));
#endif
}

void PlatformerScreen::hasEaten(float pCholesterol)
{
    if (mNetwork.get())
    {
        (*mNetwork).platformerEaten(pCholesterol);
    }
}

void PlatformerScreen::removeSprite(int pSpriteIndex)
{
    delete mSprites[pSpriteIndex];
    mSprites[pSpriteIndex] = NULL;
}

void PlatformerScreen::onVictory()
{
    if (mNetwork.get())
    {
        (*mNetwork).platformerVictory();
    }
    ScreenManager::getInstance()->popScreen();
    ScreenManager::getInstance()->pushScreen(new WinScreen(std::move(mNetwork), true));
}