#include "platformer/Player.h"

#include <SFML/Window/Keyboard.hpp>
#include <cmath>

#include "engine/GameTime.h"
#include "platformer/PlatformerCollisionLayers.h"

#include "Config.h"
#include "engine/SFMLMath.h"
#include "platformer/Collectible.h"
#include "platformer/Enemy.h"
#include "platformer/PlatformerScreen.h"

Player::Player(const sf::Vector2f& pInitPosition, float pScreenWidth, CollisionManager& pCollisionManager, PlatformerScreen& pPlatformerScreen) :
    AnimatedSprite(pInitPosition),
    mScreenWidth(pScreenWidth),
    mCollisionManager(pCollisionManager),
    mPlatformerScreen(pPlatformerScreen),
    mTimeHurt(0),
    mIsGrounded(true)
{
    addAnimation("idle", "data/platformer/Player/idle", 0.5f, true);
    play("idle", true, true);
    addAnimation("jump", "data/platformer/Player/jump", 1.0f, false);
    addAnimation("hit", "data/platformer/Player/hit", Config::PlatformerPlayer::twoArmsHitDuration, true);
    addAnimation("walk", "data/platformer/Player/walk", 0.5f, true);
    addAnimation("pant", "data/platformer/Player/down", 0.5f, true);
    sf::Rect<float> lBounds = Sprite::getSFSprite().getLocalBounds();
    mPlayerHittingAABB.setBoxSize(lBounds.width, lBounds.height);
    mPlayerHittingAABB.setBoxPosition(pInitPosition);
    mPlayerHittenAABB.setBoxSize(lBounds.width / 2, lBounds.height / 2);
    mPlayerHittenAABB.setBoxPosition(pInitPosition + sf::Vector2f(lBounds.width / 4, lBounds.height / 4));
    mGroundAABB.setBoxSize(lBounds.width, 1);
    mGroundAABB.setBoxPosition(pInitPosition + sf::Vector2f(0, lBounds.height));
    mIsGrounded = true;
    mGroundAABB.setFullCallback<Player, &Player::groundBelow>(*this);
    mPlayerHittenAABB.setCallback<Player, &Player::enemyTouched>(PlatformerCollisionLayers::Enemy, *this);
    mPlayerHittingAABB.setCallback<Player, &Player::collect>(PlatformerCollisionLayers::Collectible, *this);
    mPlayerHittingAABB.setCallback<Player, &Player::enemyHit>(PlatformerCollisionLayers::Enemy, *this);
    for(int i = 0; i < BODY_PARTS_CARD; ++i)
        mBodyState[i] = FULL;
    mCollisionManager.addBoundingBox(PlatformerCollisionLayers::PlayerBad, &mPlayerHittenAABB);
    mCollisionManager.addBoundingBox(PlatformerCollisionLayers::PlayerGood, &mPlayerHittingAABB);
    mCollisionManager.addBoundingBox(PlatformerCollisionLayers::PlayerGround, &mGroundAABB);
    mCollisionManager.addLayerCollisions(PlatformerCollisionLayers::PlayerGood, PlatformerCollisionLayers::Enemy);
    mCollisionManager.addLayerCollisions(PlatformerCollisionLayers::PlayerBad, PlatformerCollisionLayers::Enemy);
    mCollisionManager.addLayerCollisions(PlatformerCollisionLayers::PlayerGood, PlatformerCollisionLayers::Collectible);
    mCollisionManager.addLayerCollisions(PlatformerCollisionLayers::PlayerGround, PlatformerCollisionLayers::FirstPlatform);
    
    mGUI.mBodySprite.loadFrom("data/platformer/Player/gui/statut_neutre.png");
    mGUI.mIllBodyParts[LEFT_ARM].loadFrom("data/platformer/Player/gui/bras_g.png");
    mGUI.mIllBodyParts[RIGHT_ARM].loadFrom("data/platformer/Player/gui/bras_d.png");
    mGUI.mIllBodyParts[LEFT_LEG].loadFrom("data/platformer/Player/gui/jambe_g.png");
    mGUI.mIllBodyParts[RIGHT_LEG].loadFrom("data/platformer/Player/gui/jambe_d.png");
    mGUI.mIllBodyParts[HEAD].loadFrom("data/platformer/Player/gui/tete.png");
    for(int i = 0; i < NOISE_PARTICLES_COUNT; ++i)
    {
        switch(rand() % 2)
        {
        case 0: mGUI.mNoiseSprites[i].loadFrom("data/platformer/Player/gui/noise1.png"); break;
        case 1: mGUI.mNoiseSprites[i].loadFrom("data/platformer/Player/gui/noise2.png"); break;
        }
        mGUI.mNoiseSprites[i].setPosition(sf::Vector2f((float)(rand() % Config::Graphics::width), (float)(rand() % Config::Graphics::height)));
    }
    mGUI.mEnergyGaugeframe.loadFrom("data/platformer/Player/gui/jauge - coeur.png");
    mGUI.mEnergyGaugeframe.setPosition(sf::Vector2f(Config::Graphics::width - mGUI.mEnergyGaugeframe.getSFSprite().getLocalBounds().width - 2.0f, 2.0f));
    mGUI.mEnergyGaugeBase.loadFrom("data/platformer/Player/gui/jauge - rouge.png");
    mGUI.mEnergyGaugeBase.setPosition(sf::Vector2f(Config::Graphics::width - mGUI.mEnergyGaugeframe.getSFSprite().getLocalBounds().width - 2.0f, 2.0f));
    mGUI.mEnergyGaugeFull.loadFrom("data/platformer/Player/gui/jauge - noir.png");
    mGUI.mEnergyGaugeFull.setPosition(sf::Vector2f(Config::Graphics::width - mGUI.mEnergyGaugeframe.getSFSprite().getLocalBounds().width - 2.0f, 2.0f));
    mGUI.mEnergyUnit.loadFrom("data/platformer/Player/gui/kcal.png");
    mGUI.mEnergyUnit.setPosition(
        sf::Vector2f(Config::Graphics::width - mGUI.mEnergyGaugeBase.getSFSprite().getLocalBounds().width - mGUI.mEnergyUnit.getSFSprite().getLocalBounds().width,
                     4.0f));

    mEnergy = Config::PlatformerPlayer::maxEnergy;
    mIsRecovering = false;
    
    mVictoryCoolDown = -1.0f;
    
    sf::SoundBuffer* lSoundBuffer = new sf::SoundBuffer();
    lSoundBuffer->loadFromFile("data/platformer/Player/croc1.ogg");
    sf::Sound* lSound = new sf::Sound(*lSoundBuffer);
    mEatSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/croc2.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mEatSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/croc3.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mEatSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/croc4.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mEatSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/jump1.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mJumpSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/jump2.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mJumpSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/jump3.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mJumpSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/jump4.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mJumpSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/kill1.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mHurtSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/kill2.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mHurtSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/kill3.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mHurtSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/kill4.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    mHurtSounds.push_back(std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound));
    
    lSoundBuffer = new sf::SoundBuffer(); lSoundBuffer->loadFromFile("data/platformer/Player/sick_loop.ogg"); lSound = new sf::Sound(*lSoundBuffer);
    lSound->setLoop(true);
    mSickLoop = std::pair<sf::SoundBuffer*, sf::Sound*>(lSoundBuffer, lSound);
}

Player::~Player()
{
    for(unsigned int i = 0; i < mEatSounds.size(); ++i)
    {
        delete mEatSounds[i].second;
        delete mEatSounds[i].first;
    }
    for(unsigned int i = 0; i < mJumpSounds.size(); ++i)
    {
        delete mJumpSounds[i].second;
        delete mJumpSounds[i].first;
    }
    for(unsigned int i = 0; i < mHurtSounds.size(); ++i)
    {
        delete mHurtSounds[i].second;
        delete mHurtSounds[i].first;
    }
    
    delete mSickLoop.second;
    delete mSickLoop.first;
}

void Player::playEatSound()
{
    mEatSounds[rand() % mEatSounds.size()].second->play();
}

void Player::playJumpSound()
{
    mJumpSounds[rand() % mJumpSounds.size()].second->play();
}

void Player::playHurtSound()
{
    mHurtSounds[rand() % mHurtSounds.size()].second->play();
}

void Player::playSickSound()
{
    mSickLoop.second->play();
}

void Player::stopSickSound()
{
    mSickLoop.second->stop();
}

int Player::getInputMask() const
{
    int lReturnValue = 0;
    
    if(sf::Keyboard::isKeyPressed(sf::Keyboard::Right) || sf::Keyboard::isKeyPressed(sf::Keyboard::D))
        lReturnValue |= 1 << RIGHT;
    if(sf::Keyboard::isKeyPressed(sf::Keyboard::Left) || sf::Keyboard::isKeyPressed(sf::Keyboard::Q))
        lReturnValue |= 1 << LEFT;
    if(sf::Keyboard::isKeyPressed(sf::Keyboard::Z) || sf::Keyboard::isKeyPressed(sf::Keyboard::Up))
        lReturnValue |= 1 << JUMP;
    if(sf::Keyboard::isKeyPressed(sf::Keyboard::Space))
        lReturnValue |= 1 << HIT;

    return lReturnValue;
}

Player::BodyPartLife Player::getBodyPartLife(float pLife)
{
	if (pLife >= 1.f) return Player::FULL;
	if (pLife >= .75f) return Player::HIGH;
	if (pLife >= .5f) return Player::ACCEPTABLE;
	if (pLife >= .3f) return Player::LOW;
	if (pLife >= .15f) return Player::VERY_LOW;
	return Player::CRITIC;
}

void Player::updateLife(Network::Member pBodyPart, float pLife)
{
	BodyPartLife lLife = getBodyPartLife(pLife);
	if (lLife < mBodyState[pBodyPart])
	{
		mBodyState[pBodyPart] = lLife;
	}
}

void Player::update()
{
    int lArmCount = mBodyState[LEFT_ARM] + mBodyState[RIGHT_ARM];
    AnimatedSprite::update();
    for (unsigned int i = 0, l = mCollectiblesToDestroy.size() ; i < l ; ++i)
    {
        destroyCollectible(mCollectiblesToDestroy[i]);
    }
    mCollectiblesToDestroy.clear();
    for (unsigned int i = 0, l = mEnemiesToDestroy.size() ; i < l ; ++i)
    {
        destroyEnemy(mEnemiesToDestroy[i]);
    }
    mEnemiesToDestroy.clear();
    int lCurrentInputsMask = getInputMask();
    if(mVictoryCoolDown > 0.0f)
    {
        lCurrentInputsMask = 0;
    }
    else if(mIsRecovering)
    {
        mEnergy += Time::deltaTime() * Config::PlatformerPlayer::gainedEnergyPerSecondOnRecovery;
        if(mEnergy >= Config::PlatformerPlayer::recoveredEnergyOnRecovery)
        {
            mIsRecovering = false;
            stopSickSound();
        }
        else
            lCurrentInputsMask = 0;
    }

    if(mIsRecovering)
        AnimatedSprite::play("pant");
    else if(lCurrentInputsMask & (1 << HIT) && lArmCount)
    {
        mEnergy -= (float)lArmCount * 0.5f * Config::PlatformerPlayer::consumedEnergyPerSecondOnHit * Time::deltaTime();
        if(mEnergy <= 0.0f)
        {
            playSickSound();
            mEnergy = 0.0f;
            mIsRecovering = true;
        }
        mCollisionManager.addLayerCollisions(PlatformerCollisionLayers::PlayerGood, PlatformerCollisionLayers::Enemy);
        AnimatedSprite::play("hit");
    }
    else if(mIsGrounded)
    {
        mCollisionManager.removeLayerCollisions(PlatformerCollisionLayers::PlayerGood, PlatformerCollisionLayers::Enemy);
        if ((!(lCurrentInputsMask & (1 << RIGHT)) && !(lCurrentInputsMask & (1 << LEFT)))
         || ( (lCurrentInputsMask & (1 << RIGHT)) &&  (lCurrentInputsMask & (1 << LEFT))))
            AnimatedSprite::play("idle");
        else
            AnimatedSprite::play("walk");
    }
    else
    {
        mCollisionManager.removeLayerCollisions(PlatformerCollisionLayers::PlayerGood, PlatformerCollisionLayers::Enemy);
        AnimatedSprite::play("jump");
    }
    if(!mIsGrounded)
    {
        mVelocity += Config::PlatformerPlayer::gravity * Time::deltaTime();
    }
    else if((lCurrentInputsMask & (1 << JUMP)) && !(mPreviousInputsMask & (1 << JUMP)))
    {
        mEnergy -= Config::PlatformerPlayer::consumedEnergyOnJump;
        if(mEnergy <= 0.0f)
        {
            playSickSound();
            mIsRecovering = true;
            mEnergy = 0.0f;
        }
        playJumpSound();
        mIsGrounded = false;
        if (mBodyState[LEFT_LEG] && mBodyState[RIGHT_LEG])
            mVelocity.y = Config::PlatformerPlayer::twoLegsJumpImpulse;
        else if (mBodyState[LEFT_LEG] || mBodyState[RIGHT_LEG])
            mVelocity.y = Config::PlatformerPlayer::oneLegJumpImpulse;
        else
            mVelocity.y = Config::PlatformerPlayer::noLegJumpImpulse;
    }
    else if (mVelocity.y >= 0)
    {
        mVelocity.y = 0;
    }

    if (mVelocity.y < 0 || mTimeHurt > 0)
    {
        mCollisionManager.removeLayerCollisions(PlatformerCollisionLayers::PlayerGround, PlatformerCollisionLayers::Platform);
    }
    else
    {
        mCollisionManager.addLayerCollisions(PlatformerCollisionLayers::PlayerGround, PlatformerCollisionLayers::Platform);
    }

    if (mTimeHurt > 0)
    {
        mTimeHurt -= Time::deltaTime();
        mTimeHurt = std::max(mTimeHurt, 0.f);
        if (mTimeHurt == 0)
        {
            mCollisionManager.addLayerCollisions(PlatformerCollisionLayers::PlayerBad, PlatformerCollisionLayers::Enemy);
        }
    }
    
    float lHorizontalAcceleration;
    float lMaxHorizontalVelocity;
    if (mBodyState[LEFT_LEG] && mBodyState[RIGHT_LEG])
    {
        lHorizontalAcceleration = Config::PlatformerPlayer::twoLegsHorizontalAcceleration;
        lMaxHorizontalVelocity = Config::PlatformerPlayer::twoLegsMaxHorizontalVelocity;
    }
    else if (mBodyState[LEFT_LEG] || mBodyState[RIGHT_LEG])
    {
        lHorizontalAcceleration = Config::PlatformerPlayer::oneLegHorizontalAcceleration;
        lMaxHorizontalVelocity = Config::PlatformerPlayer::oneLegMaxHorizontalVelocity;
    }
    else
    {
        lHorizontalAcceleration = Config::PlatformerPlayer::noLegHorizontalAcceleration;
        lMaxHorizontalVelocity = Config::PlatformerPlayer::noLegMaxHorizontalVelocity;
    }

    if((lCurrentInputsMask & (1 << RIGHT)) || (lCurrentInputsMask & (1 << LEFT)))
    {
        mEnergy -= Config::PlatformerPlayer::consumedEnergyPerSecondOnMove * Time::deltaTime();
        if(mEnergy <= 0.0f)
        {
            playSickSound();
            mEnergy = 0.0f;
            mIsRecovering = true;
        }
    }

    if(lCurrentInputsMask & (1 << RIGHT))
    {
        AnimatedSprite::flipX(false);
        mVelocity.x += Time::deltaTime() * lHorizontalAcceleration;
    }
    if(lCurrentInputsMask & (1 << LEFT))
    {
        AnimatedSprite::flipX(true);
        mVelocity.x -= Time::deltaTime() * lHorizontalAcceleration;
    }
    if(mVelocity.x > lMaxHorizontalVelocity)
        mVelocity.x = lMaxHorizontalVelocity;
    else if(mVelocity.x < -lMaxHorizontalVelocity)
        mVelocity.x = -lMaxHorizontalVelocity;
    
    float lFrameFriction = Time::deltaTime() * (mIsGrounded ? Config::PlatformerPlayer::horizontalGroundFriction : Config::PlatformerPlayer::horizontalAirFriction);
    if(!(lCurrentInputsMask & (1 << RIGHT)) && (mVelocity.x > 0.0f))
    {
        if(mVelocity.x <= lFrameFriction)
            mVelocity.x = 0.0f;
        else
            mVelocity.x -= lFrameFriction;
    }
    if(!(lCurrentInputsMask & (1 << LEFT)) && (mVelocity.x < 0.0f))
    {
        if(mVelocity.x >= -lFrameFriction)
            mVelocity.x = 0.0f;
        else
            mVelocity.x += lFrameFriction;
    }
    sf::Vector2f lNewPosition = AnimatedSprite::getPosition() + Time::deltaTime() * mVelocity;
    if(lNewPosition.x < 0.0f)
    {
        mVelocity.x = 0.0f;
        lNewPosition.x = 0.0f;
    }
    else if(lNewPosition.x + mSprite->getLocalBounds().width > mScreenWidth)
    {
        mVelocity.x = 0.0f;
        lNewPosition.x = mScreenWidth - mSprite->getLocalBounds().width;
    }

    Player::setPosition(lNewPosition);

    if(!mBodyState[HEAD])
        for(int i = 0; i < NOISE_PARTICLES_COUNT; ++i)
        {
            sf::Vector2f lNewPosition =
                mGUI.mNoiseSprites[i].getPosition() +
                sf::Vector2f((float)((rand() % 7) - 3), (float)((rand() % 7) - 3));
            if(lNewPosition.x < 0.0f)
                lNewPosition.x = 0.0f;
            else if(lNewPosition.x > Config::Graphics::width)
                lNewPosition.x = Config::Graphics::width;
            if(lNewPosition.y < 0.0f)
                lNewPosition.y = 0.0f;
            else if(lNewPosition.y > Config::Graphics::height)
                lNewPosition.y = Config::Graphics::height;
            mGUI.mNoiseSprites[i].setPosition(lNewPosition);
        }
    
    mGUI.mEnergyGaugeFull.getSFSprite().setScale(1.0f - (mEnergy / Config::PlatformerPlayer::maxEnergy), 1.0f);

    mPreviousInputsMask = lCurrentInputsMask;
    mIsGrounded = false;
}

void Player::setPosition(sf::Vector2f const& pPosition)
{
    sf::Vector2f lOffset = pPosition - AnimatedSprite::getPosition();
    AnimatedSprite::setPosition(pPosition);
    mPlayerHittenAABB.setBoxPosition(mPlayerHittenAABB.getBoxPosition() + lOffset);
    mPlayerHittingAABB.setBoxPosition(mPlayerHittingAABB.getBoxPosition() + lOffset);
    mGroundAABB.setBoxPosition(mGroundAABB.getBoxPosition() + lOffset);
}

void Player::groundBelow(BoundingBox const& pCollider, unsigned int pLayer)
{
    mIsGrounded = true;
	if (mVelocity.y >= 0)
	{
		float lYDiff = AnimatedSprite::getPosition().y - mGroundAABB.getBoxPosition().y;
		Player::setPosition(sf::Vector2f(AnimatedSprite::getPosition().x, pCollider.getBoxRect().top + lYDiff));
	}
    if (pLayer == PlatformerCollisionLayers::FirstPlatform)
    {
        mTimeHurt = 0;
        mCollisionManager.addLayerCollisions(PlatformerCollisionLayers::PlayerBad, PlatformerCollisionLayers::Enemy);
    }
}

void Player::enemyTouched(BoundingBox const& pCollider)
{
    if (mVelocity.y < 10) mVelocity.y = 10;
    mVelocity.x = -sign(mVelocity.x) * std::max(std::abs(mVelocity.x) * 5, Config::PlatformerEnemy::speed) * 5;
    playHurtSound();
    mTimeHurt = TIME_TO_BE_HURT;
    mCollisionManager.removeLayerCollisions(PlatformerCollisionLayers::PlayerBad, PlatformerCollisionLayers::Enemy);
}

void Player::enemyHit(BoundingBox const& pCollider)
{
    Enemy* lEnemy = const_cast<Enemy*>(static_cast<Enemy const*>(&pCollider));
    if (lEnemy->tryKill(((mBodyState[LEFT_ARM] && mBodyState[RIGHT_ARM]) ? Config::PlatformerPlayer::twoArmsHitDamage : Config::PlatformerPlayer::oneArmHitDamage) * Time::deltaTime()))
    {
        mEnemiesToDestroy.push_back(lEnemy);
    }
}

void Player::collect(BoundingBox const& pCollider)
{
    mCollectiblesToDestroy.push_back(static_cast<Collectible const*>(&pCollider));
    mEnergy += static_cast<Collectible const*>(&pCollider)->getEnergyValue();
    if(mEnergy > Config::PlatformerPlayer::maxEnergy)
        mEnergy = Config::PlatformerPlayer::maxEnergy;
    playEatSound();
}

void Player::destroyCollectible(Collectible const* pCollectible)
{
    mCollisionManager.removeBoudingBox(*pCollectible, PlatformerCollisionLayers::Collectible);
    // TODO apply bonuses
    mPlatformerScreen.hasEaten(pCollectible->getCholesterolValue());
    pCollectible->removeFromList();
}

void Player::destroyEnemy(Enemy const* pEnemy)
{
    mCollisionManager.removeBoudingBox(*pEnemy, PlatformerCollisionLayers::Enemy);
    pEnemy->removeFromList();
}

void Player::draw(sf::RenderTarget* pTarget) const
{
    AnimatedSprite::draw(pTarget);

    pTarget->setView(
        sf::View(
            sf::Vector2f(0.5f * Config::Graphics::width,
                         0.5f * Config::Graphics::height),
            sf::Vector2f((float)Config::Graphics::width,
                         (float)Config::Graphics::height)));
    if(!mBodyState[HEAD])
        for(int i = 0; i < NOISE_PARTICLES_COUNT; ++i)
            mGUI.mNoiseSprites[i].draw(pTarget);
    mGUI.mBodySprite.draw(pTarget);
    for(int i = 0; i < BODY_PARTS_CARD; ++i)
	{
		mGUI.mIllBodyParts[i].getSFSprite().setColor(sf::Color(255, 255, 255, getOpacity(mBodyState[i])));
		mGUI.mIllBodyParts[i].draw(pTarget);
	}
    mGUI.mEnergyUnit.draw(pTarget);
    mGUI.mEnergyGaugeBase.draw(pTarget);
    mGUI.mEnergyGaugeFull.draw(pTarget);
    mGUI.mEnergyGaugeframe.draw(pTarget);
}

void Player::killMember(Network::Member pBodyPart)
{
    mBodyState[pBodyPart] = KILLED;
    switch (pBodyPart)
    {
    case LEFT_ARM:
    case RIGHT_ARM:
        mAnimations["hit"]->setSpeed(Config::PlatformerPlayer::oneArmHitDuration);
        break;
    }
}

uint8_t Player::getOpacity(Player::BodyPartLife pLife)
{
	switch (pLife)
	{
	case Player::FULL:
		return 0;
	case Player::HIGH:
		return 7;
	case Player::ACCEPTABLE:
		return 15;
	case Player::LOW:
		return 31;
	case Player::VERY_LOW:
		return 63;
	case Player::CRITIC:
		return 127;
	case Player::KILLED:
		return 255;
	}
	return 0;
}