#include "str/BodyGrid.h"

BodyGrid::BodyGrid()
{

}

void BodyGrid::init(int pWidth, int pHeight)
{
	mWidth = pWidth;
	mHeight = pHeight;

	mGrid = new BodyGridCell[mWidth*mHeight];
}

//-------------------------------------------------------------

void BodyGrid::fromFile(const sf::String& pFile)
{
	sf::Image img;
	img.loadFromFile(pFile+"/map.png");

	init(img.getSize().x, img.getSize().y);

	for(int i = 0; i < mWidth; i++)
	{
		for(int j = 0; j < mHeight; j++)
		{
			unsigned int idx = i*mHeight + j;
			sf::Color col = img.getPixel(i,j);

			if(col == sf::Color(0,0,0))
			{
				mGrid[idx].setType( BodyGridCell::Type::WALL );
			}
			else
			{
				mGrid[idx].setType( BodyGridCell::Type::GROUND );
			}

			mGrid[idx].setPosition(sf::Vector2f( i*32 - mWidth*0.5f*32, j*32 - mHeight*0.5f*32));
			mGrid[idx].setId(mPathfinder.addNode(mGrid[idx].getPosition()));

			mGrid[idx].setGrid(i,j,this);
		}
	}
}

//------------------------------------------------------------

sf::Vector2i BodyGrid::getSize()
{
	return sf::Vector2i(mWidth, mHeight);
}

//-------------------------------------------------------------

void BodyGrid::initPathFinder()
{
	for(int i = 0; i < mWidth; i++)
	{
		for(int j = 0; j < mHeight; j++)
		{
			unsigned int idx = i*mHeight + j;
			if(i > 0)
			{
				if(mGrid[(i-1) * mHeight + j].isPassable())
					mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[(i-1) * mHeight + j].getId());

				if(j > 0)
				{
					if(mGrid[(i-1) * mHeight + (j-1)].isPassable())
						mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[(i-1) * mHeight + (j-1)].getId());
				}

				if(j < mHeight-1)
				{
					if(mGrid[(i-1) * mHeight + (j+1)].isPassable())
						mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[(i-1) * mHeight + (j+1)].getId());
				}
			}

			//***********************************

			if(i < mWidth-1)
			{
				if(mGrid[(i+1) * mHeight + j].isPassable())
					mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[(i+1) * mHeight + j].getId());

				if(j > 0)
				{
					if(mGrid[(i+1) * mHeight + (j-1)].isPassable())
						mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[(i+1) * mHeight + (j-1)].getId());
				}

				if(j < mHeight-1)
				{
					if(mGrid[(i+1) * mHeight + (j+1)].isPassable())
						mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[(i+1) * mHeight + (j+1)].getId());
				}
			}

			//***************************************

			if(j > 0)
			{
				if(mGrid[i * mHeight + (j-1)].isPassable())
					mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[i * mHeight + (j-1)].getId());

				if(i > 0)
				{
					if(mGrid[(i-1) * mHeight + (j-1)].isPassable())
						mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[(i-1) * mHeight + (j-1)].getId());
				}

				if(i < mWidth-1)
				{
					if(mGrid[(i+1) * mHeight + (j-1)].isPassable())
						mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[(i+1) * mHeight + (j-1)].getId());
				}
			}

			//*******************************************

			if(j < mHeight-1)
			{
				if(mGrid[i * mHeight + (j+1)].isPassable())
					mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[i * mHeight + (j+1)].getId());

				if(i > 0)
				{
					if(mGrid[(i-1) * mHeight + (j+1)].isPassable())
						mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[(i-1) * mHeight + (j+1)].getId());
				}

				if(i < mWidth-1)
				{
					if(mGrid[(i+1) * mHeight + (j+1)].isPassable())
						mPathfinder.addNodeNeighbour(mGrid[idx].getId(), mGrid[(i+1) * mHeight + (j+1)].getId());
				}
			}
		}
	}
}

//-------------------------------------------------------------

void BodyGrid::update()
{
	std::list<STRObject*>::const_iterator it = mObjects.begin();

	while(it != mObjects.end())
	{
		(*it)->update();
		it++;
	}


	it = mObjToDelete.begin();
	while(it != mObjToDelete.end())
	{
		mObjects.remove(*it);
		delete *it;
		it++;
	}
	mObjToDelete.clear();
}

//--------------------------------------------------------------

void BodyGrid::draw(sf::RenderTarget* pTarget) const
{
	sf::IntRect rect = pTarget->getViewport(pTarget->getView());

	for(int i = 0; i < mWidth; i++)
	{
		for(int j = 0; j < mHeight; j++)
		{
			mGrid[i*mHeight+j].update();
			mGrid[i*mHeight+j].draw(pTarget);
		}
	}


	std::list<STRObject*>::const_iterator it = mObjects.begin();
	while(it != mObjects.end())
	{
		(*it)->draw(pTarget);
		it++;
	}
}

//---------------------------------------------------------------

BodyGridCell* BodyGrid::getAt(int i, int j)
{
	return &mGrid[i*mHeight + j];
}

//--------------------------------------------------------------

BodyGridCell* BodyGrid::getAtWorld(sf::Vector2f pWorldCase)
{
	sf::Vector2i idx = getCellIdxFromWorld(pWorldCase);

	return &mGrid[idx.x*mHeight + idx.y];
}

//--------------------------------------------------------------

void BodyGrid::registerObject(STRObject* pObject, int i, int j)
{
	pObject->setPosition(sf::Vector2f(i*32 - mWidth*0.5f*32, j*32 - mHeight*0.5f*32));

	pObject->onPlaced(this, i, j);

	mObjects.push_back(pObject);
}

//--------------------------------------------------------------

void BodyGrid::unregisterObject(STRObject* pObject)
{
	mObjToDelete.push_back(pObject);
}

//--------------------------------------------------------------

sf::Vector2i BodyGrid::getCellIdxFromWorld(sf::Vector2f pWorldPos)
{
	int i = mWidth*0.5f + pWorldPos.x / 32.0f;
	int j = mHeight*0.5f + pWorldPos.y / 32.0f;

	i = std::min(mWidth, std::max(0, i));
	j = std::min(mHeight, std::max(0, j));

	return sf::Vector2i(i,j);
}

//--------------------------------------------------------------

std::list<STRObject*>* BodyGrid::getObjects()
{
	return &mObjects;
}

//--------------------------------------------------------------

PathFinder* BodyGrid::getPathfinder()
{
	return &mPathfinder;
}

//--------------------------------------------------------------
