#include <windows.h>
#include <iostream>
#include <fstream>
#include <io.h>
#include <fcntl.h>

#include "str/STRScreen.h"

#include "engine/GameTime.h"
#include "engine/Helpers.h"
#include "engine/InputData.h"
#include "engine/SFMLMath.h"
#include "Config.h"

#include "str/units/Gatherer.h"
#include "str/units/Fighter.h"
#include "str/objects/FatPile.h"
#include "str/objects/FatFactory.h"

#include "home/HomeScreen.h"
#include "home/WinScreen.h"


STRScreen::STRScreen(std::unique_ptr<Network> pNetwork) :
	AScreen(),
    mNetwork(std::move(pNetwork)),
    mKilledMembers(0),
	mHeight(0)
{
	/*
	AllocConsole();

    HANDLE handle_out = GetStdHandle(STD_OUTPUT_HANDLE);
    int hCrt = _open_osfhandle((long) handle_out, _O_TEXT);
    FILE* hf_out = _fdopen(hCrt, "w");
    setvbuf(hf_out, NULL, _IONBF, 1);
    *stdout = *hf_out;

    HANDLE handle_in = GetStdHandle(STD_INPUT_HANDLE);
    hCrt = _open_osfhandle((long) handle_in, _O_TEXT);
    FILE* hf_in = _fdopen(hCrt, "r");
    setvbuf(hf_in, NULL, _IONBF, 128);
    *stdin = *hf_in;*/

	//*************************************************

	fromFolder("data/STR/map/base");

	mLeftPressed = sf::Mouse::isButtonPressed(sf::Mouse::Left);
	mMoveCamera = sf::Mouse::isButtonPressed(sf::Mouse::Right);

	mCurrentUnit = NULL;

	mSelectRect.setFillColor(sf::Color::Transparent);
	mSelectRect.setOutlineColor(sf::Color::Blue);
	mSelectRect.setOutlineThickness(2.0f);
	mSelectRect.setSize(sf::Vector2f(32, 32));

	mZoom = 1.0f;

	mEnergy = 50;
	mEnergyText.setPosition(0,0);

	mGoodEnergy = 0;
	mGoodEnergyText.setPosition(0, 30);

	mHeight = 0;
	mHeightText.setPosition(0, 60);
	mHeightText.setString("Opponent Progression: 0%");

	mGrasSpawn = 0;

	//*************************************************

	mMusic.openFromFile("data/STR/ggj_team_prog_2.ogg");
    mMusic.setLoop(true);
    mMusic.play();

	spawnFatUnit();

	mSpawnKey[0] = sf::Keyboard::isKeyPressed(sf::Keyboard::X);
	mSpawnKey[1] = sf::Keyboard::isKeyPressed(sf::Keyboard::C);
}

//----------------------------------------------------

void STRScreen::fromFolder(const sf::String& pFolder)
{
	mGrid.fromFile(pFolder);

	std::cout<<(pFolder+"/info.txt").toAnsiString()<<std::endl;

	std::ifstream t((pFolder+"/info.txt").toAnsiString());

	if(!t.is_open())
		return;

	std::string str;

	t.seekg(0, std::ios::end);   
	str.reserve(t.tellg());
	t.seekg(0, std::ios::beg);

	str.assign((std::istreambuf_iterator<char>(t)),
				std::istreambuf_iterator<char>());

	str.erase(remove_if(str.begin(), str.end(), isspace), str.end());
	std::vector<std::string> split = strSplit(str, ';');

	for(int i = 0; i < split.size(); ++i)
	{
		std::string data = split[i];

		std::vector<std::string> infos = strSplit(split[i], '=');

		if(infos.size() >= 2)
		{
			std::string variable = infos[0];
			std::string value = infos[1];

			if(variable == "factory")
			{
				value.erase(value.begin());
				value.erase(value.begin()+(value.length()-1));

				std::vector<std::string> vec = strSplit(value, ',');

				float x = stringToFloat(vec[0]);
				float y = stringToFloat(vec[1]);

				float type = stringToFloat(vec[2]);

				FatFactory* obj = new FatFactory();
				obj->setMember((int)type);
				mGrid.registerObject(obj, x, y);
			}
			else if(variable == "spawn")
			{
				value.erase(value.begin());
				value.erase(value.begin()+(value.length()-1));

				std::vector<std::string> vec = strSplit(value, ',');

				float x = stringToFloat(vec[0]);
				float y = stringToFloat(vec[1]);

				mSpawn = sf::Vector2i(x,y);
			}
			else if(variable == "fatSpawn")
			{
				value.erase(value.begin());
				value.erase(value.begin()+(value.length()-1));

				std::vector<std::string> vec = strSplit(value, ',');

				float x = stringToFloat(vec[0]);
				float y = stringToFloat(vec[1]);
				float w = stringToFloat(vec[2]);
				float h = stringToFloat(vec[3]);

				mFatSpawn.push_back(sf::IntRect(x,y,w,h));
			}
			else if(variable == "leucocyteSpawn")
			{
				value.erase(value.begin());
				value.erase(value.begin()+(value.length()-1));

				std::vector<std::string> vec = strSplit(value, ',');

				float x = stringToFloat(vec[0]);
				float y = stringToFloat(vec[1]);
				float w = stringToFloat(vec[2]);
				float h = stringToFloat(vec[3]);

				mLeucocyteSpawn.push_back(sf::IntRect(x,y,w,h));
			}
		}
	}


	//-------

	mGrid.initPathFinder();
}

//--------------------------------------

void STRScreen::update()
{
    if (mNetwork.get())
    {
        Network& lNetwork = *mNetwork;
        Network::Message lMsg;
        while (lNetwork.getMessage(lMsg))
        {
            if (lMsg.isForRtsPlayer())
            {
                if (lMsg.isEat())
                {
                    float lCholesterol = lMsg.getCholesterol();
                    if (lCholesterol > 0)
                    {
                        mEnergy += lCholesterol * Config::Str::energyPerCholesterol;
                        mGoodEnergy += lCholesterol * Config::Str::goodEnergyPerCholesterol;
                        mGrasSpawn += lCholesterol * Config::Str::grasPerCholesterol;
                    }
                    else
                    {
                        mEnergy += lCholesterol * Config::Str::energyPerNegativeCholesterol;
                        mGoodEnergy += lCholesterol * Config::Str::goodEnergyPerNegativeCholesterol;
                        mGrasSpawn += lCholesterol * Config::Str::grasPerNegativeCholesterol;
                    }
                }
                else if (lMsg.isHeight())
                {
					mHeight = lMsg.getHeight();
					std::stringstream mStm;
					mStm << "Opponent Progression: " << (int)(mHeight * 100) << "%";
					mHeightText.setString(mStm.str());
                }
                else if (lMsg.isWin())
                {
                    ScreenManager::getInstance()->popScreen();
                    ScreenManager::getInstance()->pushScreen(new WinScreen(std::move(mNetwork), false));
                    return;
                }
            }
        }
        if (lNetwork.isDisconnected())
        {
            ScreenManager::getInstance()->popScreen();
            ScreenManager::getInstance()->pushScreen(new HomeScreen(Config::Graphics::width, Config::Graphics::height));
            return;
        }
    }
	cameraHandling();
	mGrid.update();

	manageInput();

	mEnergy += Time::deltaTime() * 0.5f;
	//mGoodEnergy -= Time::deltaTime();
	mGrasSpawn -= Time::deltaTime();

	if(mEnergy > 9999.9f)
	{
		mEnergy = 9999.9f;
	}

	if(mGoodEnergy <= 0)
	{
		if(mEnnemies.size() < 10)
		{
			spawnLeucocyte();
			mGoodEnergy = 20;
		}
		else
		{
			mGoodEnergy = 0;
		}
	}

	if(mGrasSpawn <= 0)
	{
		spawnFat();
		mGrasSpawn = 15;
	}

	mEnergyText.setString("LIPID : " +intToString((int)mEnergy));
	mGoodEnergyText.setString("Next Leucocyte : "+intToString((int)mGoodEnergy));

	std::list<Unit*> toRemove;
	std::list<Unit*>::iterator it = mUnits.begin();
	while(it != mUnits.end())
	{
		(*it)->update();

		Fighter* f = dynamic_cast<Fighter*>(*it);
		if(f)
		{
			f->checkLeucocyte(mEnnemies);
		}

		if((*it)->isDead() && (*it)->getReference() <= 0)
			toRemove.push_back(*it);

		++it;
	}

	std::list<Leucocyte*> toRemoveE;
	std::list<Leucocyte*>::iterator itE = mEnnemies.begin();
	while(itE != mEnnemies.end())
	{
		(*itE)->lookForTarget(mUnits);
		(*itE)->update();

		if((*itE)->isDead( )&& (*itE)->getReference() <= 0)
			toRemoveE.push_back(*itE);

		++itE;
	}

	
	it = toRemove.begin();
	while(it != toRemove.end())
	{
		mUnits.remove(*it);

		if(*it == mCurrentUnit)
			mCurrentUnit = NULL;

		delete *it;

		++it;
	}

	itE = toRemoveE.begin();
	while(itE != toRemoveE.end())
	{
		mEnnemies.remove(*itE);

		delete *itE;

		++itE;
	}

    if (mKilledMembers >= Network::HEART)
    {
        if (mNetwork.get())
        {
            (*mNetwork).strVictory();
        }
        ScreenManager::getInstance()->popScreen();
        ScreenManager::getInstance()->pushScreen(new WinScreen(std::move(mNetwork), true));
        return;
    }
}


//------------------------------------

void STRScreen::draw(sf::RenderTarget* pTarget)
{
	pTarget->setView(mView);
	pTarget->clear();
	mGrid.draw(pTarget);

	std::list<Unit*>::iterator it = mUnits.begin();
	while(it != mUnits.end())
	{
		(*it)->draw(pTarget);
		it++;
	}

	std::list<Leucocyte*>::iterator itE = mEnnemies.begin();
	while(itE != mEnnemies.end())
	{
		(*itE)->draw(pTarget);
		++itE;
	}

	if(mCurrentUnit != NULL)
	{
		mSelectRect.setPosition(mCurrentUnit->getPosition());
		pTarget->draw(mSelectRect);
	}

	sf::View v = pTarget->getView();

	pTarget->setView(sf::View(sf::Vector2f(Config::Graphics::width * 0.5, Config::Graphics::height*0.5f), sf::Vector2f(Config::Graphics::width, Config::Graphics::height)));
	pTarget->draw(mEnergyText);
	pTarget->draw(mGoodEnergyText);
	if (mNetwork.get())
	{
		pTarget->draw(mHeightText);
	}

	pTarget->setView(v);
}

//-------------------------------------

void STRScreen::spawnFatUnit()
{
	if(mEnergy < 50)
		return;

	mEnergy -= 50;

	Unit* u = new Gatherer();
	
	u->setGrid(&mGrid);
	u->setCase(mSpawn.x, mSpawn.y);
	u->setPosition(mGrid.getAt(mSpawn.x, mSpawn.y)->getPosition());
	mUnits.push_back(u);
}

//-------------------------------------

void STRScreen::spawnFatFighter()
{
	if(mEnergy < 100)
		return;

	mEnergy -= 100;

	Unit* u = new Fighter();
	
	u->setGrid(&mGrid);
	u->setCase(mSpawn.x, mSpawn.y);
	u->setPosition(mGrid.getAt(mSpawn.x, mSpawn.y)->getPosition());
	mUnits.push_back(u);
}

//-------------------------------------

void STRScreen::spawnFat()
{
	STRObject* o = new FatPile();

	sf::IntRect rect = mFatSpawn[rand()%mFatSpawn.size()];
	
	int i = rect.left + rand()% rect.width;
	int j = rect.top + rand() % rect.height;

	mGrid.registerObject(o, i, j);
}

//-------------------------------------

void STRScreen::spawnLeucocyte()
{

	sf::IntRect rect = mLeucocyteSpawn[rand()%mLeucocyteSpawn.size()];
	sf::Vector2i pos = sf::Vector2i(rect.left + rand()%rect.width, rect.top + rand()%rect.height);

	Leucocyte* l = new Leucocyte();

	l->setGrid(&mGrid);
	l->setCase(pos.x, pos.y);
	l->setPosition(mGrid.getAt(pos.x, pos.y)->getPosition());
	mEnnemies.push_back(l);
}

//-------------------------------------

void STRScreen::manageInput()
{
	if(!sf::Mouse::isButtonPressed(sf::Mouse::Left))
	{
		if(mLeftPressed)
		{
			if(!checkSelection() && mCurrentUnit)
			{
				sf::Vector2i pos = sf::Mouse::getPosition(*ScreenManager::getInstance()->getWin());
				sf::Vector2f world = screenToWorld(pos, ScreenManager::getInstance()->getWin());

				sf::Vector2i cell = mGrid.getCellIdxFromWorld(world);

				if(mGrid.getAt(cell.x, cell.y)->isPassable())
				{
					mCurrentUnit->gotoCase(cell.x, cell.y);
				}
			}
		}

		mLeftPressed = false;
	}
	else
	{
		mLeftPressed = true;
	}

	//-------------------------------------------------------

	if(InputData::getInstance()->mouseWheelDelta > 0)
	{
		if(mZoom > 1)
		{
			mZoom *= 1/1.5f;
			mView.zoom(1/1.5f);
		}
	}
	else if(InputData::getInstance()->mouseWheelDelta < 0)
	{
		if(mZoom < 2.25f)
		{
			mZoom *= 1.5f;
			mView.zoom(1.5f);
		}
	}

	if(sf::Keyboard::isKeyPressed(sf::Keyboard::X))
	{
		if(!mSpawnKey[0])
		{
			spawnFatUnit();
		}

		mSpawnKey[0] = true;
	}
	else
	{
		mSpawnKey[0] = false;
	}

	//------------------------------------------------

	if(sf::Keyboard::isKeyPressed(sf::Keyboard::C))
	{
		if(!mSpawnKey[1])
		{
			spawnFatFighter();
		}

		mSpawnKey[1] = true;
	}
	else
	{
		mSpawnKey[1] = false;
	}

	//------------------------------------------------
}

//-------------------------------------

void STRScreen::cameraHandling()
{
	sf::Window* win = mManager->getWin();
	sf::Vector2i pos = sf::Mouse::getPosition(*win);

	sf::Vector2f actualCenter = mView.getCenter();

	float speed = 100;


	if(!sf::Mouse::isButtonPressed(sf::Mouse::Right))
	{
		if(pos.x < 20)
		{
			actualCenter.x += -speed*Time::deltaTime();
		}
		else if(pos.x > win->getSize().x - 20)
		{
			actualCenter.x += speed*Time::deltaTime();
		}

		if(pos.y < 20)
		{
			actualCenter.y += -speed*Time::deltaTime();
		}
		else if(pos.y > win->getSize().y - 20)
		{
			actualCenter.y += speed*Time::deltaTime();
		}

		mMoveCamera = false;
	}
	else
	{
		sf::Vector2i diff = pos - sf::Vector2i(win->getSize().x*0.5f, win->getSize().y*0.5f);

		float sqrtMag = sqrMagnitude(sf::Vector2f(diff.x, diff.y));
		float purcentage = sqrtMag / sqrMagnitude(sf::Vector2f(win->getSize().x*0.5, win->getSize().y*0.5f));

		actualCenter += purcentage * speed * mZoom * 2 * mView.getViewport().width * Time::deltaTime()*normalize(sf::Vector2f(diff.x, diff.y));

		mMoveCamera = true;
	}

	//--------------

	if(sf::Keyboard::isKeyPressed(sf::Keyboard::R))
	{
		actualCenter = sf::Vector2f(0,0);
	}

	mView.setCenter(actualCenter);
}

//---------------------------------------------------------------------------------------------

bool STRScreen::checkSelection()
{
	sf::Vector2i pos = sf::Mouse::getPosition(*ScreenManager::getInstance()->getWin());
	sf::Vector2f world = screenToWorld(pos, ScreenManager::getInstance()->getWin());

	std::list<Unit*>::iterator it = mUnits.begin();

	while(it != mUnits.end())
	{
		if((*it)->getSFSprite().getGlobalBounds().contains(world))
		{
			mCurrentUnit = (*it);
			return true;
		}
		it++;
	}

	return false;
}

void STRScreen::killMember(Network::Member pMember)
{
    if (mNetwork.get())
    {
        (*mNetwork).strKillMember(pMember);
    }
    ++mKilledMembers;
}

void STRScreen::sendLife(Network::Member pMember, float pLife)
{
	if (mNetwork.get())
	{
		mNetwork->strSendMemberLife(pMember, pLife);
	}
}
