#include "str/Unit.h"
#include "engine/SFMLMath.h"
#include "engine/GameTime.h"

sf::SoundBuffer* Unit::sBuffer = NULL;

Unit::Unit()
	:AnimatedSprite(sf::Vector2f(0,0))
{
	if(sBuffer == NULL)
	{
		sBuffer = new sf::SoundBuffer();
		sBuffer->loadFromFile("data/STR/sounds/hit.ogg");
	}

	sound.setBuffer(*sBuffer);

	mMustMove = false;
	mAtTarget = true;

	mHealthBar.setFillColor(sf::Color(255, 0, 0));
	mHealthBar.setOutlineColor(sf::Color(0,0,0));
	mHealthBar.setOutlineThickness(1.0);

	mCaseChanged = false;
	mBlocked = false;

	mDead = false;

	mReferenced = 0;

	mSinceLastHurt = 0;
}

//--------------------------------------------------------

void Unit::setGrid(BodyGrid* pGrid)
{
	mGrid = pGrid;
	mCase = NULL;
}

//--------------------------------------------------------

void Unit::setCase(int i, int j)
{
	if(mCase != NULL)
	{
		mCase->unregisterUnit(this);
	}

	mCase = mGrid->getAt(i,j);
	mCase->registerUnit(this);

	mCaseChanged = true;
}

//--------------------------------------------------------

void Unit::gotoCase(int i, int j)
{
	mTarget.x = i;
	mTarget.y = j;

	if(mAtTarget)
	{
		mGrid->getPathfinder()->computePath(mCase->getId(), mGrid->getAt(i,j)->getId(), mPath);

		if(mPath.size() > 0)
		{
			mCurrentPathIdx = 0;

			mAtTarget = false;
			mCurrentToward = mPath[0];
		}
	}
	else
	{
		sf::Vector2i idx = mGrid->getCellIdxFromWorld(mCurrentToward);
		mGrid->getPathfinder()->computePath(mGrid->getAt(idx.x, idx.y)->getId(), mGrid->getAt(i,j)->getId(), mPath);

		mCurrentPathIdx = -1;
	}

	mMustMove = true;
}

//--------------------------------------------------------

void Unit::update()
{
	mCaseChanged = false;

	AnimatedSprite::update();

	if(mSinceLastHurt > 0)
	{
		mSinceLastHurt -= Time::deltaTime();
	}

	//**

	float size = mHealth/mMaxHealth * 32;
	mHealthBar.setSize(sf::Vector2f(4, size));
	mHealthBar.setPosition(getPosition().x + 28, getPosition().y + (32 - size));

	//**

	if(mDead)
		return;

	checkObjectUse();

	if(!mAtTarget && !mBlocked)
	{
		sf::Vector2f targetPos = mCurrentToward;//mGrid->getAt(mTarget.x, mTarget.y)->getPosition();

		setPosition(moveToward(getPosition(), targetPos, mSpeed * Time::deltaTime()));

		if(getPosition().x == targetPos.x && getPosition().y == targetPos.y)
		{
			sf::Vector2i idx = mGrid->getCellIdxFromWorld(getPosition());

			setCase(idx.x, idx.y);

			mCurrentPathIdx++;
			if(mCurrentPathIdx == mPath.size())
			{
				mAtTarget = true;
			}
			else
			{
				mCurrentToward = mPath[mCurrentPathIdx];
			}
			
		}
	}

}

//--------------------------------------------------------

void Unit::draw(sf::RenderTarget* pTarget) const
{
	AnimatedSprite::draw(pTarget);

	if(mHealth != mMaxHealth)
		pTarget->draw(mHealthBar);
}

//--------------------------------------------------------

void Unit::checkObjectUse()
{
	std::list<STRObject*>* objs = mGrid->getObjects();

	std::list<STRObject*>::iterator it = objs->begin();

	while(it != objs->end())
	{
		float dist = sqrMagnitude(((*it)->getPosition() + (*it)->getCenter()) - getPosition());

		if(dist <= (*it)->getRange() * (*it)->getRange())
		{
			(*it)->use(this);
		}

		it++;
	}
}

//---------------------------------------------------------

bool Unit::caseChanged()
{
	return mCaseChanged;
}

//---------------------------------------------------------

void Unit::block(bool pBlock)
{
	mBlocked = pBlock;
}

bool Unit::isBlocked() const
{
	return mBlocked;
}

//----------------------------------------------------------

void Unit::hurt(float pDamage)
{
	mHealth = std::max(0.0f, mHealth - pDamage);

	if(mSinceLastHurt <= 0)
	{
		 sound.play();
		 mSinceLastHurt = 1.0f;
	}

	if(mHealth <= 0.0001f)
	{
		die();
	}
}

//----------------------------------------------------------

void Unit::die()
{
	mDead = true;
}

bool Unit::isDead() const
{
	return mDead;
}

//----------------------------------------------------------

void Unit::reference()
{
	mReferenced += 1;;
}

void Unit::unreference()
{
	mReferenced -= 1;
}

int Unit::getReference() const
{
	return mReferenced;
}