#include "engine/GameTime.h"

#include "str/objects/FatFactory.h"
#include "str/units/Gatherer.h"
#include "str/STRScreen.h"
#include "engine/ScreenManager.h"
#include "Config.h"

FatFactory::FatFactory()
{
	mFatMax = 100;
	mFat = 0;
	mTimeToSendLife = ((float)rand() / (float)RAND_MAX) * TIME_TO_SEND_LIFE;

	loadFrom("data/STR/Body/factory.png");

	mCenter = sf::Vector2f(mSprite->getTextureRect().width * 0.5f, mSprite->getTextureRect().height * 0.5f);
	mRange = 120.0f;

	mFatBar.setSize(sf::Vector2f(10, 0));
	mFatBar.setFillColor(sf::Color(255,255,0));

	mCellFatified = 0;
	mWorking = true;
}

//---------------------------------

void FatFactory::addFat(float pAmount)
{
    mFat = std::min(mFat + pAmount * Config::Str::multiplierGatherAmount, mFatMax);
}

//----------------------------------

void FatFactory::setMember(int pNum)
{
	switch(pNum)
	{
	case 0:
		mMember = Network::HEAD;
		break;
	case 1:
		mMember = Network::LEFT_ARM;
		break;
	case 2:
		mMember = Network::RIGHT_ARM;
		break;
	case 3:
		mMember = Network::LEFT_LEG;
		break;
	case 4:
		mMember = Network::RIGHT_LEG;
		break;
	default:
		break;
	}
}

//----------------------------------

void FatFactory::onPlaced(BodyGrid* pGrid, int i, int j)
{
	setGrid(pGrid);

	mIndex.x = i;
	mIndex.y = j;

	for(int k = 0; k < 4; ++k)
	{
		for(int l = 0; l < 3; ++l)
		{
			mGrid->getAt(i+k,j+l)->setPassable(false);
		}
	}

	sf::Vector2i idx = mGrid->getCellIdxFromWorld(getPosition());
	for(int k = -1; k < 5; ++k)
	{
		for(int l = -1; l < 4; ++l)
		{
			if(idx.x + k > 0 && idx.x + k < mGrid->getSize().x &&
				idx.y + l > 0 && idx.y + l < mGrid->getSize().y)
			{
				addToFatification(idx.x + k, idx.y + l);
			}
		}
	}
}

//----------------------------------

void FatFactory::addToFatification(int i, int j)
{
	BodyGridCell* c = mGrid->getAt(i,j);

	if(!c->isVisited() && c->isPassable())
	{
		c->setVisited(true);
		mToAdd.push_back(c);
	}
}

//----------------------------------

void FatFactory::use(Unit* pUnit)
{
	Gatherer* g = dynamic_cast<Gatherer*>(pUnit);

	if(g)
	{
		float remainingfat = g->getFat();

		if(remainingfat <= 0)
			return;

		float amount = Time::deltaTime() * 3.0f;

		amount = std::min(amount, remainingfat);

		g->addFat(-amount);
		addFat(amount);
	}
}

//------------------------------------------------------

void FatFactory::update()
{
	float size = mFat/mFatMax * 32;
	mFatBar.setSize(sf::Vector2f(10, size));

	mFatBar.setPosition(getPosition().x, getPosition().y + (32 - size));

	if(!mWorking)
		return;

	mTimeToSendLife -= Time::deltaTime();
	if (mTimeToSendLife < 0)
	{
		mTimeToSendLife = TIME_TO_SEND_LIFE;
		float lLife = 1 - mCellFatified / 200.f;
		if (lLife < 0) lLife = 0;
		if (lLife > 1) lLife = 1;
		static_cast<STRScreen*>(ScreenManager::getInstance()->peekScreen())->sendLife(mMember, lLife);
	}

	if(mFat > 0.0001f)
	{
        float amount = std::min(Time::deltaTime() * Config::Str::multiplierGrasSpeed, mFat);

		mFat -= amount;

		fatification(amount * 0.1f);
	}
}

//------------------------------------------------------

void FatFactory::fatification(float pAmount)
{
	std::list<BodyGridCell*>::iterator it = mToFatify.begin();
	sf::Vector2i size = mGrid->getSize();

	while(it != mToFatify.end())
	{
		BodyGridCell* c = (*it);

		c->increaseFatification(pAmount);

		if(c->getFat() > 0.99f)
		{
			int i = c->getIndex().x;
			int j = c->getIndex().y;

			for(int k = -1; k <= 1; ++k)
			{
				for(int l = -1; l <= 1; ++l)
				{
					if(k == 0 && l == 0)
						continue;

					if(i+k >= 0 && i+k < size.x && j+l >= 0 && j+l < size.y)
					{
						if(!mGrid->getAt((i+k), (j+l))->isVisited())
						{
							addToFatification(i+k, j+l);
						}
					}
				}
			}

			mToRemove.push_back(c);
		}

		++it;
	}

	//***

	it = mToRemove.begin();
	while(it != mToRemove.end())
	{
		mToFatify.remove(*it);
		mCellFatified++;
		++it;
	}
	
	mToRemove.clear();
	

	it = mToAdd.begin();
	while(it != mToAdd.end())
	{
		mToFatify.push_back(*it);
		++it;
	}

	mToAdd.clear();

	if(mCellFatified > 200)
	{
		fatificationComplete();
	}
}

//------------------------------------------------------

void FatFactory::fatificationComplete()
{
    static_cast<STRScreen*>(ScreenManager::getInstance()->peekScreen())->killMember(mMember);
	mWorking = false;
}

//------------------------------------------------------

void FatFactory::draw(sf::RenderTarget* pTarget) const
{
    if (mWorking)
    {
	    STRObject::draw(pTarget);
	    pTarget->draw(mFatBar);
    }
}